import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, Heart, Users, Briefcase, Sparkles, Award, Clock, Star, User } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookSession.css'

const BookSession = () => {
  const navigate = useNavigate()
  const [selectedStyle, setSelectedStyle] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const styles = [
    { 
      id: 'portrait', 
      name: 'Портрет', 
      icon: Camera,
      color: '#FF6B9D',
      description: 'Индивидуальные портретные съёмки',
      gradient: 'linear-gradient(135deg, #FF6B9D 0%, #FFB6C1 100%)',
      duration: '1-2 часа',
      price: 'от 5000₽'
    },
    { 
      id: 'family', 
      name: 'Семейная', 
      icon: Users,
      color: '#C44569',
      description: 'Семейные фотосессии',
      gradient: 'linear-gradient(135deg, #C44569 0%, #FF6B9D 100%)',
      duration: '2-3 часа',
      price: 'от 8000₽'
    },
    { 
      id: 'lovestory', 
      name: 'Love Story', 
      icon: Heart,
      color: '#FFB6C1',
      description: 'Романтические съёмки',
      gradient: 'linear-gradient(135deg, #FFB6C1 0%, #FF6B9D 100%)',
      duration: '2-4 часа',
      price: 'от 10000₽'
    },
    { 
      id: 'brand', 
      name: 'Бренд', 
      icon: Briefcase,
      color: '#9B59B6',
      description: 'Брендовые фотосессии',
      gradient: 'linear-gradient(135deg, #9B59B6 0%, #C44569 100%)',
      duration: '3-5 часов',
      price: 'от 15000₽'
    }
  ]

  const photographers = [
    { name: 'Анна Петрова', experience: '8 лет', rating: 5, specialty: 'Love Story, Портрет' },
    { name: 'Михаил Соколов', experience: '12 лет', rating: 5, specialty: 'Бренд, Портрет' },
    { name: 'Елена Волкова', experience: '6 лет', rating: 5, specialty: 'Семейная, Love Story' },
    { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Бренд, Портрет' }
  ]

  const advantages = [
    { icon: Award, title: 'Профессиональное оборудование', text: 'Работаем только с топовым фотооборудованием' },
    { icon: Clock, title: 'Гибкий график', text: 'Подстраиваемся под ваше удобное время' },
    { icon: Star, title: 'Опытные фотографы', text: 'Все наши специалисты с многолетним опытом' },
    { icon: Camera, title: 'Быстрая обработка', text: 'Готовые фото в течение 3-5 дней' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedStyle && selectedDate) {
      const savedSessions = localStorage.getItem('photoSessions')
      const sessions = savedSessions ? JSON.parse(savedSessions) : []
      
      const styleData = styles.find(s => s.id === selectedStyle)
      const randomPhotographer = photographers[Math.floor(Math.random() * photographers.length)]
      
      const newSession = {
        id: Date.now(),
        photographer: randomPhotographer.name,
        theme: styleData.name,
        date: selectedDate,
        rating: 0
      }

      sessions.push(newSession)
      localStorage.setItem('photoSessions', JSON.stringify(sessions))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedStyle('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-session-page">
      <div className="container">
        <motion.section 
          className="book-hero"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="book-icon-wrapper"
            initial={{ scale: 0, rotate: 180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Sparkles size={72} strokeWidth={2} className="book-hero-icon" />
          </motion.div>
          <h1 className="book-title">Записаться на фотосессию</h1>
          <p className="book-subtitle">
            Выберите стиль съёмки и удобную дату для создания незабываемых фотографий
          </p>
        </motion.section>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="booking-form-wrapper">
            <form className="booking-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Camera size={22} strokeWidth={2.5} className="label-icon" />
                  Выберите стиль съёмки
                </label>
                <div className="styles-grid">
                  {styles.map((style) => {
                    const IconComponent = style.icon
                    return (
                      <motion.button
                        key={style.id}
                        type="button"
                        className={`style-card ${selectedStyle === style.id ? 'selected' : ''}`}
                        onClick={() => setSelectedStyle(style.id)}
                        whileHover={{ scale: 1.05, rotate: 2 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedStyle === style.id ? style.color : 'rgba(255, 107, 157, 0.2)',
                          background: selectedStyle === style.id ? (style.color + '15') : 'rgba(255, 255, 255, 0.95)'
                        }}
                      >
                        <div className="style-icon-wrapper" style={{ background: style.color + '20' }}>
                          <IconComponent size={40} strokeWidth={2.5} color={style.color} />
                        </div>
                        <div className="style-content">
                          <h3 className="style-name">{style.name}</h3>
                          <p className="style-description">{style.description}</p>
                          <div className="style-meta">
                            <span className="style-duration">{style.duration}</span>
                            <span className="style-price">{style.price}</span>
                          </div>
                        </div>
                        {selectedStyle === style.id && (
                          <motion.div 
                            className="style-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            style={{ background: style.gradient }}
                          >
                            <CheckCircle size={24} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="date" className="form-label">
                  <Calendar size={22} strokeWidth={2.5} className="label-icon" />
                  Дата фотосессии
                </label>
                <input
                  type="date"
                  id="date"
                  className="form-input"
                  value={selectedDate}
                  onChange={(e) => setSelectedDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedStyle || !selectedDate}
                whileHover={{ scale: 1.02, y: -3 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на фотосессию</span>
                <CheckCircle size={22} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={28} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно записались на фотосессию!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="photographers-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Наши фотографы</h2>
          <div className="photographers-grid">
            {photographers.map((photographer, index) => (
              <motion.div
                key={photographer.name}
                className="photographer-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ scale: 1.05, rotate: 2 }}
              >
                <div className="photographer-avatar">
                  <User size={40} strokeWidth={2} />
                </div>
                <h3 className="photographer-name">{photographer.name}</h3>
                <p className="photographer-experience">{photographer.experience} опыта</p>
                <div className="photographer-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={16} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                  ))}
                </div>
                <p className="photographer-specialty">{photographer.specialty}</p>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="advantages-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Почему выбирают нас</h2>
          <div className="advantages-grid">
            {advantages.map((advantage, index) => {
              const IconComponent = advantage.icon
              return (
                <motion.div
                  key={advantage.title}
                  className="advantage-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ translateY: -8, boxShadow: '0 12px 32px rgba(255, 107, 157, 0.2)' }}
                >
                  <div className="advantage-icon">
                    <IconComponent size={32} strokeWidth={2.5} color="#FF6B9D" />
                  </div>
                  <h3 className="advantage-title">{advantage.title}</h3>
                  <p className="advantage-text">{advantage.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookSession
